<?php

namespace App\Models;

use App\Constants\Status;
use App\Traits\GlobalStatus;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;

class Property extends Model
{
    use  GlobalStatus;

    protected $casts = [
        'working_schedule' => 'object',
        'amenities'        => 'object',
        'tags'             => 'object',
        'videos'           => 'object',
        'day_off'          => 'object',
    ];

    public function category()
    {
        return $this->belongsTo(Category::class);
    }
    public function country()
    {
        return $this->belongsTo(Country::class);
    }
    public function city()
    {
        return $this->belongsTo(City::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }
    public function images()
    {
        return $this->hasMany(PropertyImage::class);
    }

    public function statusBadge(): Attribute
    {
        return new Attribute(function () {
            $html = '';
            if ($this->status == Status::PROPERTY_APPROVED) {
                $html = '<span class="badge badge--success">' . trans("Approved") . '</span>';
            } elseif ($this->status == Status::PROPERTY_DISABLED) {
                $html = '<span class="badge badge--dark">' . trans("Disabled") . '</span>';
            } elseif ($this->status == Status::PROPERTY_PENDING) {
                $html = '<span class="badge badge--warning">' . trans("Pending") . '</span>';
            } elseif ($this->status == Status::PROPERTY_REJECTED) {
                $html = '<span class="badge badge--danger">' . trans("Rejected") . '</span>';
            }
            return $html;
        });
    }

    public function featuredBadge(): Attribute
    {
        return new Attribute(function () {
            $html = '';
            if ($this->is_featured == Status::PROPERTY_FEATURED) {
                $html = '<span class="badge badge--primary">' . trans("Yes") . '</span>';
            } else {
                $html = '<span class="badge badge--warning">' . trans("No") . '</span>';
            }
            return $html;
        });
    }

    // SCOPES
    public function scopeBuy($query)
    {
        return $query->where('property_status', Status::BUY);
    }

    public function scopeRent($query)
    {
        return $query->where('property_status', Status::RENT);
    }

    public function scopePending($query)
    {
        return $query->where('status', Status::PROPERTY_PENDING);
    }

    public function scopeApproved($query)
    {
        return $query->where('status', Status::PROPERTY_APPROVED);
    }
    public function scopeRejected($query)
    {
        return $query->where('status', Status::PROPERTY_REJECTED);
    }

    public function scopeDisabled($query)
    {
        return $query->where('status', Status::PROPERTY_DISABLED);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', Status::PROPERTY_FEATURED);
    }

    public function scopePublishable($query)
    {
        return $query->approved()->whereHas('user.activePlan', Status::PROPERTY_DISABLED);
    }
}
